/**
 * 文件排序 Hook
 * 提供文件排序功能
 */
(function() {
    'use strict';
    
    const { ref, computed } = typeof Vue !== 'undefined' ? Vue : {};
    
    if (!Vue) {
        console.error('Vue is not loaded');
        return;
    }
    
    window.useFileSort = function() {
        // 排序选项
        const sortOptions = [
            { value: 'date-desc', label: '最新优先', field: 'timestamp', order: 'desc' },
            { value: 'date-asc', label: '最早优先', field: 'timestamp', order: 'asc' },
            { value: 'name-asc', label: '名称 A-Z', field: 'name', order: 'asc' },
            { value: 'name-desc', label: '名称 Z-A', field: 'name', order: 'desc' },
            { value: 'size-desc', label: '大小 大-小', field: 'size', order: 'desc' },
            { value: 'size-asc', label: '大小 小-大', field: 'size', order: 'asc' }
        ];
        
        // 当前排序方式
        const currentSort = ref('date-desc');
        
        // 获取当前排序配置
        const getCurrentSortConfig = computed(() => {
            return sortOptions.find(opt => opt.value === currentSort.value) || sortOptions[0];
        });
        
        // 排序文件列表
        const sortFiles = (files, sortValue) => {
            if (!files || !Array.isArray(files)) return [];
            
            // 如果提供了 sortValue，使用它；否则使用当前的排序方式
            const sortConfig = sortValue 
                ? sortOptions.find(opt => opt.value === sortValue) || sortOptions[0]
                : getCurrentSortConfig.value;
            
            const { field, order } = sortConfig;
            
            return [...files].sort((a, b) => {
                let aValue = a[field];
                let bValue = b[field];
                
                // 处理时间戳
                if (field === 'timestamp') {
                    aValue = aValue || 0;
                    bValue = bValue || 0;
                }
                
                // 处理文件名（字符串）
                if (field === 'name') {
                    aValue = (aValue || '').toLowerCase();
                    bValue = (bValue || '').toLowerCase();
                }
                
                // 处理文件大小（数字）
                if (field === 'size') {
                    aValue = aValue || 0;
                    bValue = bValue || 0;
                }
                
                // 排序比较
                if (order === 'asc') {
                    if (aValue < bValue) return -1;
                    if (aValue > bValue) return 1;
                    return 0;
                } else {
                    if (aValue > bValue) return -1;
                    if (aValue < bValue) return 1;
                    return 0;
                }
            });
        };
        
        // 设置排序方式
        const setSort = (sortValue) => {
            if (sortOptions.find(opt => opt.value === sortValue)) {
                currentSort.value = sortValue;
            }
        };
        
        return {
            sortOptions,
            currentSort,
            getCurrentSortConfig,
            sortFiles,
            setSort
        };
    };
})();

