/**
 * 文件卡片组件
 */
(function() {
    'use strict';
    
    const { computed } = typeof Vue !== 'undefined' ? Vue : {};
    
    if (!Vue || !window.useFileTypes || !window.useFileUtils) {
        console.error('Required dependencies not loaded');
        return;
    }
    
    window.FileCard = {
        props: {
            file: {
                type: Object,
                required: true
            },
            viewMode: {
                type: String,
                default: 'grid' // grid | list
            }
        },
        emits: ['click', 'download', 'preview', 'delete'],
        template: `
            <div 
                :class="['file-card', viewMode === 'list' ? 'file-card-list' : 'file-card-grid']"
                @click="handleClick">
                <!-- 网格视图 -->
                <div v-if="viewMode === 'grid'" class="file-card-grid-content">
                    <div class="file-card-thumbnail">
                        <div class="file-card-thumbnail-overlay">
                            <div class="file-card-checkbox">
                                <el-checkbox />
                            </div>
                            <div class="file-card-actions" @click.stop>
                                <el-button 
                                    type="primary" 
                                    circle 
                                    size="small"
                                    @click="handlePreview"
                                    title="预览">
                                    <el-icon><View /></el-icon>
                                </el-button>
                                <el-button 
                                    type="success" 
                                    circle 
                                    size="small"
                                    @click="handleDownload"
                                    title="下载">
                                    <el-icon><Download /></el-icon>
                                </el-button>
                                <el-button 
                                    type="danger" 
                                    circle 
                                    size="small"
                                    @click="handleDelete"
                                    title="删除">
                                    <el-icon><Delete /></el-icon>
                                </el-button>
                            </div>
                        </div>
                        <!-- 图片或视频显示缩略图 -->
                        <img
                            v-if="(fileType === 'image' || fileType === 'video') && thumbnailUrl"
                            :src="thumbnailUrl"
                            :alt="fileName"
                            class="file-card-image"
                            @error="handleImageError" />
                        <!-- 加载中占位图 -->
                        <div v-else-if="file.isLoading" class="file-card-icon-wrapper" :style="{ backgroundColor: '#e5e7eb' }">
                            <div class="file-card-loading">
                                <el-icon :size="24" color="#9ca3af"><Loading /></el-icon>
                                <span class="loading-text">加载中...</span>
                            </div>
                        </div>
                        <!-- 其他文件类型显示图标 -->
                        <div v-else class="file-card-icon-wrapper" :style="{ backgroundColor: typeConfig.color + '15' }">
                            <el-icon :size="48" :color="typeConfig.color" class="file-card-icon">
                                <component :is="typeConfig.icon" />
                            </el-icon>
                        </div>
                    </div>
                    <div class="file-card-info">
                        <div class="file-card-name-wrapper">
                            <el-icon class="file-card-name-icon" :size="16" :color="typeConfig.color">
                                <component :is="typeConfig.icon" />
                            </el-icon>
                            <div class="file-card-name" :title="fileName">{{ fileName }}</div>
                        </div>
                        <div class="file-card-meta">
                            <el-icon class="meta-icon" :size="12"><Document /></el-icon>
                            <span>{{ formatFileSize(fileSize) }}</span>
                            <span class="meta-separator">·</span>
                            <el-icon class="meta-icon" :size="12"><Clock /></el-icon>
                            <span>{{ formatDate(file.timestamp) }}</span>
                        </div>
                    </div>
                </div>
                
                <!-- 列表视图（表格样式） -->
                <div v-else class="file-card-list-content file-card-table-row">
                    <div class="file-card-table-cell file-card-table-name">
                        <el-checkbox />
                        <div class="file-card-list-icon" :style="{ backgroundColor: typeConfig.color + '15' }">
                            <el-icon :size="24" :color="typeConfig.color">
                                <component :is="typeConfig.icon" />
                            </el-icon>
                        </div>
                        <div class="file-card-name" :title="fileName">{{ fileName }}</div>
                    </div>
                    <div class="file-card-table-cell file-card-table-size">
                        <span>{{ file.type === 'folder' ? '-' : formatFileSize(fileSize) }}</span>
                    </div>
                    <div class="file-card-table-cell file-card-table-type">
                        <span>{{ file.type === 'folder' ? '文件夹' : typeConfig.label }}</span>
                    </div>
                    <div class="file-card-table-cell file-card-table-time">
                        <span>{{ file.timestamp ? formatDate(file.timestamp) : '-' }}</span>
                    </div>
                </div>
            </div>
        `,
        setup(props, { emit }) {
            const { getFileType, getFileTypeConfig } = window.useFileTypes();
            const { formatFileSize, formatDate } = window.useFileUtils();

            // 兼容获取文件名
            const fileName = computed(() => {
                return props.file.file_name || props.file.name || '';
            });

            // 兼容获取文件大小
            const fileSize = computed(() => {
                return props.file.file_size || props.file.size || 0;
            });

            // 兼容获取文件扩展名
            const fileExtension = computed(() => {
                return props.file.file_extension || props.file.fileExtension || '';
            });

            // 兼容获取缩略图
            const thumbnailUrl = computed(() => {
                return props.file.thumbnailUrl || props.file.thumbnail || '';
            });

            // 获取文件类型
            const fileType = computed(() => {
                // 如果是文件夹，返回 'folder'
                if (props.file.type === 'folder') {
                    return 'folder';
                }
                // 优先使用 file.file_type 或 file.category
                if (props.file.file_type) {
                    return props.file.file_type.toLowerCase();
                }
                if (props.file.category) {
                    return props.file.category;
                }
                // 根据扩展名判断
                const ext = fileExtension.value.toLowerCase().replace('.', '');
                if (ext) {
                    const extMap = {
                        'jpg': 'image', 'jpeg': 'image', 'png': 'image', 'gif': 'image', 'bmp': 'image', 'webp': 'image',
                        'mp4': 'video', 'avi': 'video', 'mov': 'video', 'wmv': 'video', 'flv': 'video', 'mkv': 'video',
                        'mp3': 'audio', 'wav': 'audio', 'flac': 'audio', 'aac': 'audio',
                        'pdf': 'pdf',
                        'zip': 'archive', 'rar': 'archive', '7z': 'archive',
                        'js': 'code', 'jsx': 'code', 'ts': 'code', 'html': 'code', 'css': 'code', 'json': 'code'
                    };
                    if (extMap[ext]) {
                        return extMap[ext];
                    }
                }
                // 根据 content_type 判断
                if (props.file.content_type) {
                    const contentType = props.file.content_type.toLowerCase();
                    if (contentType.includes('image')) return 'image';
                    if (contentType.includes('video')) return 'video';
                    if (contentType.includes('audio')) return 'audio';
                    if (contentType.includes('pdf')) return 'pdf';
                }
                return getFileType(fileName.value, props.file.mimeType);
            });
            
            // 获取类型配置
            const typeConfig = computed(() => {
                // 如果是文件夹，返回文件夹配置
                if (props.file.type === 'folder') {
                    return { label: '文件夹', icon: 'Folder', color: '#F59E0B' };
                }
                
                // 根据文件类型返回对应的图标配置
                const type = fileType.value;
                const iconMap = {
                    'image': { label: '图片', icon: 'Picture', color: '#67C23A' },
                    'video': { label: '视频', icon: 'VideoPlay', color: '#409EFF' },
                    'audio': { label: '音频', icon: 'Headset', color: '#E6A23C' },
                    'document': { label: '文档', icon: 'Document', color: '#F56C6C' },
                    'spreadsheet': { label: '表格', icon: 'Grid', color: '#67C23A' },
                    'pdf': { label: 'PDF', icon: 'Document', color: '#F56C6C' },
                    'archive': { label: '压缩包', icon: 'Box', color: '#909399' },
                    'code': { label: '代码', icon: 'Document', color: '#606266' },
                    'other': { label: '其他', icon: 'QuestionFilled', color: '#909399' }
                };
                
                if (iconMap[type]) {
                    return iconMap[type];
                }
                
                return getFileTypeConfig(type) || { label: '未知', icon: 'QuestionFilled', color: '#909399' };
            });
            
            // 处理图片加载错误
            const handleImageError = (event) => {
                // 如果缩略图加载失败，显示默认图标
                event.target.style.display = 'none';
            };
            
            // 处理点击
            const handleClick = () => {
                emit('click', props.file);
            };
            
            // 处理预览
            const handlePreview = () => {
                emit('preview', props.file);
            };
            
            // 处理下载
            const handleDownload = () => {
                emit('download', props.file);
            };
            
            // 处理删除
            const handleDelete = () => {
                emit('delete', props.file);
            };
            
            return {
                fileName,
                fileSize,
                fileExtension,
                thumbnailUrl,
                fileType,
                typeConfig,
                formatFileSize,
                formatDate,
                handleClick,
                handlePreview,
                handleDownload,
                handleDelete,
                handleImageError
            };
        }
    };
})();

