/**
 * 文件详情组件
 */
(function() {
    'use strict';
    
    const { ref, computed } = typeof Vue !== 'undefined' ? Vue : {};
    
    if (!Vue) {
        console.error('Vue is not loaded');
        return;
    }
    
    window.FileDetails = {
        props: {
            selectedFile: {
                type: Object,
                default: null
            },
            files: {
                type: Array,
                default: () => []
            },
            expanded: {
                type: Boolean,
                default: true
            },
            currentPath: {
                type: String,
                default: '全部文件'
            }
        },
        emits: ['close', 'preview', 'download', 'delete', 'path-click'],
        template: `
            <div class="file-details-container" :class="{ 'is-collapsed': !expanded }">
                <!-- 文件详情面板 -->
                <div v-if="expanded" class="file-details-panel">
                    <!-- 头部 -->
                    <div class="file-details-header">
                        <h3 class="file-details-title">文件详情</h3>
                        <button 
                            class="file-details-collapse-btn"
                            @click="$emit('close')"
                            title="收起">
                            <el-icon><ArrowRight /></el-icon>
                            <span class="collapse-text">收起</span>
                        </button>
                    </div>
                    
                    <!-- 内容区域 -->
                    <div class="file-details-content">
                        <!-- 未选中文件时的空状态 -->
                        <div v-if="!selectedFile" class="file-details-empty">
                            <div class="file-details-empty-icons">
                                <div class="empty-icon-folder">
                                    <el-icon :size="80" color="#d1d5db"><Folder /></el-icon>
                                </div>
                                <div class="empty-icon-doc-1">
                                    <el-icon :size="64" color="#d1d5db"><Document /></el-icon>
                                </div>
                                <div class="empty-icon-doc-2">
                                    <el-icon :size="48" color="#d1d5db"><Document /></el-icon>
                                </div>
                            </div>
                            <p class="file-details-empty-text">选中文件/文件夹，查看详情</p>
                        </div>
                        
                        <!-- 选中文件时的详情信息 -->
                        <div v-else class="file-details-info">
                            <!-- 文件预览区域 -->
                            <div class="file-details-preview">
                                <!-- 有缩略图时显示缩略图 -->
                                <div v-if="shouldShowThumbnail" class="file-details-image">
                                    <img :src="previewUrl" :alt="fileName" @error="handleImageError" />
                                </div>
                                <!-- 没有缩略图时显示图标 -->
                                <div v-else class="file-details-icon-wrapper" :style="{ backgroundColor: typeConfig.color + '15' }">
                                    <el-icon :size="64" :color="typeConfig.color">
                                        <component :is="typeConfig.icon" />
                                    </el-icon>
                                </div>
                            </div>
                            
                            <!-- 文件元数据 -->
                            <div class="file-details-meta">
                                <!-- 文件名 -->
                                <div class="file-details-meta-item file-details-name-item">
                                    <span class="meta-label">文件名</span>
                                    <span class="meta-value file-name-value" :title="fileName">{{ fileName }}</span>
                                </div>
                                
                                <!-- 创建时间 -->
                                <div class="file-details-meta-item">
                                    <span class="meta-label">创建时间</span>
                                    <span class="meta-value">{{ formatDate(createTime) }}</span>
                                </div>
                                
                                <!-- 最后修改时间 -->
                                <div class="file-details-meta-item">
                                    <span class="meta-label">最后修改</span>
                                    <span class="meta-value">{{ formatDate(modifiedTime) }}</span>
                                </div>
                                
                                <!-- 文件格式 -->
                                <div class="file-details-meta-item">
                                    <span class="meta-label">文件格式</span>
                                    <span class="meta-value">{{ fileFormat }}</span>
                                </div>
                                
                                <!-- 文件大小 -->
                                <div class="file-details-meta-item">
                                    <span class="meta-label">文件大小</span>
                                    <span class="meta-value">{{ formatFileSize(fileSize) }}</span>
                                </div>
                                
                                <!-- 所在目录 -->
                                <div class="file-details-meta-item">
                                    <span class="meta-label">所在目录</span>
                                    <span class="meta-value path-value">
                                        <span 
                                            class="path-link" 
                                            @click="$emit('path-click')"
                                            v-if="currentPath">
                                            {{ currentPath }}
                                        </span>
                                        <span v-else>全部文件</span>
                                        <el-icon class="path-icon"><FolderOpened /></el-icon>
                                    </span>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        `,
        setup(props, { emit }) {
            const { getFileType, getFileTypeConfig } = window.useFileTypes ? window.useFileTypes() : { getFileType: () => null, getFileTypeConfig: () => ({ label: '未知', icon: 'Document', color: '#6b7280' }) };
            const { formatFileSize: formatFileSizeUtil, formatDate: formatDateUtil } = window.useFileUtils ? window.useFileUtils() : { formatFileSize: () => '0 B', formatDate: () => '-' };
            
            // 获取文件名
            const fileName = computed(() => {
                if (!props.selectedFile) return '';
                return props.selectedFile.file_name || props.selectedFile.name || '未命名文件';
            });
            
            // 获取文件大小
            const fileSize = computed(() => {
                if (!props.selectedFile) return 0;
                return props.selectedFile.file_size || props.selectedFile.size || 0;
            });
            
            // 获取创建时间
            const createTime = computed(() => {
                if (!props.selectedFile) return null;
                return props.selectedFile.timestamp || props.selectedFile.created_at || null;
            });
            
            // 获取修改时间
            const modifiedTime = computed(() => {
                if (!props.selectedFile) return null;
                return props.selectedFile.modified || props.selectedFile.updated_at || props.selectedFile.timestamp || null;
            });
            
            // 获取文件格式（扩展名）
            const fileFormat = computed(() => {
                if (!props.selectedFile) return '-';
                const ext = props.selectedFile.file_extension || props.selectedFile.fileExtension || '';
                if (ext) {
                    return ext.replace('.', '').toLowerCase();
                }
                // 从文件名提取扩展名
                const name = fileName.value;
                const lastDot = name.lastIndexOf('.');
                if (lastDot > 0 && lastDot < name.length - 1) {
                    return name.substring(lastDot + 1).toLowerCase();
                }
                return '-';
            });
            
            // 判断是否为图片或视频
            const isImageOrVideo = computed(() => {
                if (!props.selectedFile) return false;
                
                // 方法1: 通过 file_type 判断
                const fileType = (props.selectedFile.file_type || '').toLowerCase();
                if (fileType === 'image' || fileType === 'video') {
                    return true;
                }
                
                // 方法2: 通过 content_type 判断
                if (props.selectedFile.content_type) {
                    const contentType = props.selectedFile.content_type.toLowerCase();
                    if (contentType.includes('image/') || contentType.includes('video/')) {
                        return true;
                    }
                }
                
                // 方法3: 通过文件扩展名判断
                const format = fileFormat.value.toLowerCase();
                const imageExts = ['jpg', 'jpeg', 'png', 'gif', 'bmp', 'webp', 'svg', 'heic', 'heif'];
                const videoExts = ['mp4', 'avi', 'mov', 'wmv', 'flv', 'mkv', 'webm', 'm4v'];
                return imageExts.includes(format) || videoExts.includes(format);
            });
            
            // 判断是否为图片
            const isImage = computed(() => {
                if (!props.selectedFile) return false;
                const fileType = (props.selectedFile.file_type || '').toLowerCase();
                const format = fileFormat.value.toLowerCase();
                const imageFormats = ['jpg', 'jpeg', 'png', 'gif', 'bmp', 'webp', 'svg', 'heic', 'heif'];
                return fileType === 'image' || imageFormats.includes(format);
            });
            
            // 获取预览URL
            const previewUrl = computed(() => {
                if (!props.selectedFile) return '';
                // 优先使用 thumbnailUrl
                if (props.selectedFile.thumbnailUrl) {
                    return props.selectedFile.thumbnailUrl;
                }
                // 其次使用 thumbnail
                if (props.selectedFile.thumbnail) {
                    return props.selectedFile.thumbnail;
                }
                // 如果是图片或视频，使用 originalUrl
                if (isImageOrVideo.value && props.selectedFile.originalUrl) {
                    return props.selectedFile.originalUrl;
                }
                return '';
            });
            
            // 判断是否应该显示缩略图（参考 FileList 的处理方式）
            const shouldShowThumbnail = computed(() => {
                if (!props.selectedFile) return false;
                
                // 检查是否有缩略图URL
                const hasThumbnail = !!(props.selectedFile.thumbnailUrl || props.selectedFile.thumbnail || (isImageOrVideo.value && props.selectedFile.originalUrl));
                
                // 必须是图片或视频类型，且有缩略图URL
                return hasThumbnail && isImageOrVideo.value;
            });
            
            // 处理图片加载错误
            const handleImageError = (event) => {
                // 如果缩略图加载失败，隐藏图片，显示图标
                event.target.style.display = 'none';
            };
            
            // 获取文件类型
            const fileType = computed(() => {
                if (!props.selectedFile) return null;
                if (props.selectedFile.type === 'folder') {
                    return 'folder';
                }
                const fileTypeValue = (props.selectedFile.file_type || '').toLowerCase();
                if (fileTypeValue) {
                    return fileTypeValue;
                }
                if (window.useFileTypes) {
                    return getFileType(fileName.value, props.selectedFile.mimeType);
                }
                return null;
            });
            
            // 获取类型配置
            const typeConfig = computed(() => {
                if (!props.selectedFile) {
                    return { label: '未知', icon: 'Document', color: '#6b7280' };
                }
                
                if (props.selectedFile.type === 'folder') {
                    return { label: '文件夹', icon: 'Folder', color: '#F59E0B' };
                }
                
                // 根据文件类型返回对应的图标配置
                const type = fileType.value;
                const iconMap = {
                    'image': { label: '图片', icon: 'Picture', color: '#67C23A' },
                    'video': { label: '视频', icon: 'VideoPlay', color: '#409EFF' },
                    'audio': { label: '音频', icon: 'Headset', color: '#E6A23C' },
                    'document': { label: '文档', icon: 'Document', color: '#F56C6C' },
                    'spreadsheet': { label: '表格', icon: 'Grid', color: '#67C23A' },
                    'pdf': { label: 'PDF', icon: 'Document', color: '#F56C6C' },
                    'archive': { label: '压缩包', icon: 'Box', color: '#909399' },
                    'code': { label: '代码', icon: 'Document', color: '#606266' },
                    'other': { label: '其他', icon: 'QuestionFilled', color: '#909399' }
                };
                
                if (iconMap[type]) {
                    return iconMap[type];
                }
                
                if (window.useFileTypes) {
                    return getFileTypeConfig(type) || { label: '未知', icon: 'QuestionFilled', color: '#909399' };
                }
                
                return { label: '未知', icon: 'QuestionFilled', color: '#909399' };
            });
            
            // 格式化文件大小
            const formatFileSize = (bytes) => {
                if (!bytes || bytes === 0) return '0 B';
                return formatFileSizeUtil(bytes);
            };
            
            // 格式化日期
            const formatDate = (timestamp) => {
                if (!timestamp) return '-';
                return formatDateUtil(timestamp);
            };
            
            return {
                fileName,
                fileSize,
                createTime,
                modifiedTime,
                fileFormat,
                isImage,
                isImageOrVideo,
                previewUrl,
                shouldShowThumbnail,
                fileType,
                typeConfig,
                formatFileSize,
                formatDate,
                handleImageError
            };
        }
    };
})();

