/**
 * 文件类型管理 Hook
 * 提供文件类型分类、图标、颜色等功能
 */
(function() {
    'use strict';
    
    // 从全局 Vue 获取必要的 API
    const { ref, computed } = typeof Vue !== 'undefined' ? Vue : {};
    
    if (!Vue) {
        console.error('Vue is not loaded');
        return;
    }
    
    window.useFileTypes = function() {
        // 文件类型定义
        const fileTypes = {
            image: {
                label: '图片',
                icon: 'Picture',
                color: '#67C23A',
                extensions: ['jpg', 'jpeg', 'png', 'gif', 'bmp', 'webp', 'svg', 'ico'],
                mimeTypes: ['image/jpeg', 'image/png', 'image/gif', 'image/bmp', 'image/webp']
            },
            video: {
                label: '视频',
                icon: 'VideoPlay',
                color: '#409EFF',
                extensions: ['mp4', 'avi', 'mov', 'wmv', 'flv', 'mkv', 'webm', 'm4v'],
                mimeTypes: ['video/mp4', 'video/avi', 'video/quicktime', 'video/x-ms-wmv', 'video/x-flv', 'video/x-matroska', 'video/webm']
            },
            audio: {
                label: '音频',
                icon: 'Headset',
                color: '#E6A23C',
                extensions: ['mp3', 'wav', 'flac', 'aac', 'ogg', 'wma', 'm4a'],
                mimeTypes: ['audio/mpeg', 'audio/wav', 'audio/flac', 'audio/aac', 'audio/ogg', 'audio/x-ms-wma']
            },
            document: {
                label: '文档',
                icon: 'Document',
                color: '#F56C6C',
                extensions: [ 'doc', 'docx', 'txt', 'rtf', 'odt'],
                mimeTypes: ['image/svg+xml', 'application/msword', 'application/vnd.openxmlformats-officedocument.wordprocessingml.document', 'text/plain', 'application/rtf']
            },
            spreadsheet: {
                label: '表格',
                icon: 'Grid',
                color: '#67C23A',
                extensions: ['xls', 'xlsx', 'csv', 'ods'],
                mimeTypes: ['application/vnd.ms-excel', 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet', 'text/csv']
            },
            presentation: {
                label: '演示文稿',
                icon: 'DataBoard',
                color: '#E6A23C',
                extensions: ['pdf','ppt', 'pptx', 'odp'],
                mimeTypes: ['application/pdf;binary','application/vnd.ms-powerpoint', 'application/vnd.openxmlformats-officedocument.presentationml.presentation']
            },
            archive: {
                label: '压缩包',
                icon: 'Box',
                color: '#909399',
                extensions: ['zip', 'rar', '7z', 'tar', 'gz', 'bz2'],
                mimeTypes: ['application/zip', 'application/x-rar-compressed', 'application/x-7z-compressed', 'application/x-tar', 'application/gzip']
            },
            code: {
                label: '代码',
                icon: 'Document',
                color: '#606266',
                extensions: ['js', 'ts', 'html', 'css', 'json', 'xml', 'py', 'java', 'cpp', 'c'],
                mimeTypes: ['text/javascript', 'text/typescript', 'text/html', 'text/css', 'application/json', 'application/xml', 'text/x-python']
            },
            other: {
                label: '其他',
                icon: 'Document',
                color: '#909399',
                extensions: [],
                mimeTypes: []
            }
        };
        
        // 当前选中的文件类型
        const selectedFileType = ref('all');
        
        // 获取文件类型
        const getFileType = (fileName, mimeType) => {
          
            
            // 先通过扩展名匹配
            for (const [type, config] of Object.entries(fileTypes)) {
                if (type === 'other') continue;
                if (config.extensions.includes(fileName)) {
                    return type;
                }
            }
            
            // 再通过 MIME 类型匹配
            if (mimeType) {
                for (const [type, config] of Object.entries(fileTypes)) {
                    if (type === 'other') continue;
                    if (config.mimeTypes.some(mime => mimeType.includes(mime.split('/')[1]))) {
                        return type;
                    }
                }
            }
            
            return 'other';
        };
        
        // 获取文件类型配置
        const getFileTypeConfig = (type) => {
            return fileTypes[type] || fileTypes.other;
        };
        
        // 文件类型列表（用于筛选）
        const fileTypeList = computed(() => {
            return [
                { value: 'all', label: '全部文件', icon: 'Folder', color: '#409EFF' },
                ...Object.entries(fileTypes).map(([key, config]) => ({
                    value: key,
                    label: config.label,
                    icon: config.icon,
                    color: config.color
                }))
            ];
        });
        
        // 筛选文件
        const filterFilesByType = (files, type) => {
            if (type === 'all') return files;
            return files.filter(file => {
                
                const fileType = getFileType(file.name, file.mimeType);
                return fileType === type;
            });
        };
        
        return {
            fileTypes,
            selectedFileType,
            getFileType,
            getFileTypeConfig,
            fileTypeList,
            filterFilesByType
        };
    };
})();





