# IDDisk 组件文档

## 组件结构

IDDisk 采用组件化开发方式，所有组件位于 `src/components` 目录下。

### Hooks（工具函数）

#### 1. `useFileTypes.js`
文件类型管理 Hook，提供文件类型分类、图标、颜色等功能。

**功能：**
- 定义文件类型（图片、视频、音频、文档、表格、演示文稿、压缩包、代码等）
- 根据文件名和 MIME 类型识别文件类型
- 提供文件类型筛选功能

**使用示例：**
```javascript
const { getFileType, getFileTypeConfig, fileTypeList, filterFilesByType } = useFileTypes();
```

#### 2. `useFileSort.js`
文件排序 Hook，提供文件排序功能。

**功能：**
- 支持按日期、名称、大小排序
- 支持升序/降序
- 提供排序选项配置

**使用示例：**
```javascript
const { sortOptions, currentSort, sortFiles, setSort } = useFileSort();
```

#### 3. `useFileUtils.js`
文件工具函数 Hook，提供文件相关的工具函数。

**功能：**
- 格式化文件大小
- 格式化日期
- 获取文件扩展名
- 获取文件名（不含扩展名）

**使用示例：**
```javascript
const { formatFileSize, formatDate, getFileExtension } = useFileUtils();
```

### 组件

#### 1. `FileTypeFilter.js`
文件类型筛选组件，用于筛选不同类型的文件。

**Props：**
- `modelValue` (String): 当前选中的文件类型，默认 'all'

**Events：**
- `update:modelValue`: 文件类型改变时触发
- `change`: 文件类型改变时触发

#### 2. `FileSort.js`
文件排序组件，用于选择排序方式。

**Props：**
- `modelValue` (String): 当前排序方式，默认 'date-desc'

**Events：**
- `update:modelValue`: 排序方式改变时触发
- `change`: 排序方式改变时触发

#### 3. `FileCard.js`
文件卡片组件，用于显示单个文件信息。

**Props：**
- `file` (Object): 文件对象，必需
- `viewMode` (String): 视图模式，'grid' 或 'list'，默认 'grid'

**Events：**
- `click`: 文件点击时触发
- `download`: 下载文件时触发
- `preview`: 预览文件时触发
- `delete`: 删除文件时触发

**文件对象结构：**
```javascript
{
    id: String,           // 文件ID
    name: String,         // 文件名
    size: Number,         // 文件大小（字节）
    timestamp: Number,    // 时间戳
    mimeType: String,     // MIME 类型
    url: String,          // 文件URL
    thumbnail: String     // 缩略图URL（可选）
}
```

#### 4. `FileList.js`
文件列表组件，用于显示文件列表。

**Props：**
- `files` (Array): 文件列表，默认 []
- `selectedFileType` (String): 选中的文件类型，默认 'all'
- `sortBy` (String): 排序方式，默认 'date-desc'
- `viewMode` (String): 视图模式，'grid' 或 'list'，默认 'grid'
- `loading` (Boolean): 加载状态，默认 false

**Events：**
- `file-click`: 文件点击时触发
- `file-download`: 下载文件时触发
- `file-preview`: 预览文件时触发
- `file-delete`: 删除文件时触发
- `upload`: 上传文件时触发

#### 5. `FileToolbar.js`
文件工具栏组件，包含文件类型筛选、排序、视图切换等功能。

**Props：**
- `selectedFileType` (String): 选中的文件类型
- `sortBy` (String): 排序方式
- `viewMode` (String): 视图模式
- `selectedCount` (Number): 选中的文件数量

**Events：**
- `update:selectedFileType`: 文件类型改变时触发
- `update:sortBy`: 排序方式改变时触发
- `update:viewMode`: 视图模式改变时触发
- `upload`: 上传文件时触发
- `refresh`: 刷新文件列表时触发

#### 6. `DiskLayout.js`
网盘主布局组件，整合所有子组件。

**Props：**
- `files` (Array): 文件列表
- `loading` (Boolean): 加载状态

**Events：**
- `upload`: 上传文件时触发
- `refresh`: 刷新文件列表时触发
- `file-click`: 文件点击时触发
- `file-download`: 下载文件时触发
- `file-preview`: 预览文件时触发
- `file-delete`: 删除文件时触发

### 样式文件

#### `disk-styles.css`
网盘组件的样式文件，包含所有组件的样式定义。

**特点：**
- 响应式设计，支持移动端
- 使用 Tailwind CSS 类名
- 与 Element Plus 样式兼容

## 使用方式

### 1. 在 HTML 中引入

```html
<!-- 先引入 Hooks -->
<script src="src/components/useFileTypes.js"></script>
<script src="src/components/useFileSort.js"></script>
<script src="src/components/useFileUtils.js"></script>

<!-- 再引入组件（按依赖顺序） -->
<script src="src/components/FileTypeFilter.js"></script>
<script src="src/components/FileSort.js"></script>
<script src="src/components/FileCard.js"></script>
<script src="src/components/FileList.js"></script>
<script src="src/components/FileToolbar.js"></script>
<script src="src/components/DiskLayout.js"></script>

<!-- 引入样式 -->
<link rel="stylesheet" href="src/components/disk-styles.css" />
```

### 2. 在 Vue 组件中使用

```javascript
const App = {
    components: {
        DiskLayout: window.DiskLayout
    },
    template: `
        <DiskLayout
            :files="files"
            :loading="loading"
            @upload="handleUpload"
            @refresh="handleRefresh"
            @file-click="handleFileClick"
            @file-download="handleFileDownload"
            @file-preview="handleFilePreview"
            @file-delete="handleFileDelete" />
    `,
    setup() {
        const files = ref([]);
        const loading = ref(false);
        
        // 实现各种处理方法
        const handleUpload = () => { /* ... */ };
        const handleRefresh = () => { /* ... */ };
        // ...
        
        return {
            files,
            loading,
            handleUpload,
            handleRefresh,
            // ...
        };
    }
};
```

## 待实现功能

以下功能已预留接口，需要在主应用中实现：

1. **文件上传** (`handleDiskUpload`)
   - 实现文件选择
   - 实现文件上传到区块链
   - 更新文件列表

2. **文件列表刷新** (`handleDiskRefresh`)
   - 从 API 获取文件列表
   - 更新文件数据

3. **文件下载** (`handleFileDownload`)
   - 实现文件下载功能

4. **文件预览** (`handleFilePreview`)
   - 实现文件预览功能（图片、视频、PDF等）

5. **文件删除** (`handleFileDelete`)
   - 实现文件删除功能

## 文件类型支持

当前支持的文件类型：

- **图片**: jpg, jpeg, png, gif, bmp, webp, svg, ico
- **视频**: mp4, avi, mov, wmv, flv, mkv, webm, m4v
- **音频**: mp3, wav, flac, aac, ogg, wma, m4a
- **文档**: pdf, doc, docx, txt, rtf, odt
- **表格**: xls, xlsx, csv, ods
- **演示文稿**: ppt, pptx, odp
- **压缩包**: zip, rar, 7z, tar, gz, bz2
- **代码**: js, ts, html, css, json, xml, py, java, cpp, c

## 响应式设计

组件支持响应式布局：

- **桌面端**: 网格视图显示 5 列
- **平板**: 网格视图显示 3-4 列
- **手机**: 网格视图显示 1-2 列，列表视图优化显示

## 注意事项

1. 所有组件依赖 Vue 3 和 Element Plus
2. 组件使用 Composition API
3. 样式使用 Tailwind CSS 和自定义 CSS
4. 文件数据需要符合指定的对象结构
5. 图标使用 Element Plus 图标组件





